<?php

/*                                                                        *
 * This script is backported from the FLOW3 package "TYPO3.Fluid".        *
 *                                                                        *
 * It is free software; you can redistribute it and/or modify it under    *
 * the terms of the GNU Lesser General Public License, either version 3   *
 *  of the License, or (at your option) any later version.                *
 *                                                                        *
 * The TYPO3 project - inspiring people to share!                         *
 *                                                                        */

/**
 * Testcase for TagBuilder
 *
 */
class Tx_Fluid_Tests_Unit_Core_ViewHelper_TagBuilderTest extends Tx_Extbase_Tests_Unit_BaseTestCase {

	/**
	 * @test
	 */
	public function constructorSetsTagName() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder('someTagName');
		$this->assertEquals('someTagName', $tagBuilder->getTagName());
	}

	/**
	 * @test
	 */
	public function constructorSetsTagContent() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder('', '<some text>');
		$this->assertEquals('<some text>', $tagBuilder->getContent());
	}

	/**
	 * @test
	 */
	public function setContentDoesNotEscapeValue() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder();
		$tagBuilder->setContent('<to be escaped>', FALSE);
		$this->assertEquals('<to be escaped>', $tagBuilder->getContent());
	}

	/**
	 * @test
	 */
	public function hasContentReturnsTrueIfTagContainsText() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder('', 'foo');
		$this->assertTrue($tagBuilder->hasContent());
	}

	/**
	 * @test
	 */
	public function hasContentReturnsFalseIfContentIsNull() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder();
		$tagBuilder->setContent(NULL);
		$this->assertFalse($tagBuilder->hasContent());
	}

	/**
	 * @test
	 */
	public function hasContentReturnsFalseIfContentIsAnEmptyString() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder();
		$tagBuilder->setContent('');
		$this->assertFalse($tagBuilder->hasContent());
	}

	/**
	 * @test
	 */
	public function renderReturnsEmptyStringByDefault() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder();
		$this->assertEquals('', $tagBuilder->render());
	}

	/**
	 * @test
	 */
	public function renderReturnsSelfClosingTagIfNoContentIsSpecified() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder('tag');
		$this->assertEquals('<tag />', $tagBuilder->render());
	}

	/**
	 * @test
	 */
	public function contentCanBeRemoved() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder('tag', 'some content');
		$tagBuilder->setContent(NULL);
		$this->assertEquals('<tag />', $tagBuilder->render());
	}

	/**
	 * @test
	 */
	public function renderReturnsOpeningAndClosingTagIfNoContentIsSpecifiedButForceClosingTagIsTrue() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder('tag');
		$tagBuilder->forceClosingTag(TRUE);
		$this->assertEquals('<tag></tag>', $tagBuilder->render());
	}

	/**
	 * @test
	 */
	public function attributesAreProperlyRendered() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder('tag');
		$tagBuilder->addAttribute('attribute1', 'attribute1value');
		$tagBuilder->addAttribute('attribute2', 'attribute2value');
		$tagBuilder->addAttribute('attribute3', 'attribute3value');
		$this->assertEquals('<tag attribute1="attribute1value" attribute2="attribute2value" attribute3="attribute3value" />', $tagBuilder->render());
	}

	/**
	 * @test
	 */
	public function attributeValuesAreEscapedByDefault() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder('tag');
		$tagBuilder->addAttribute('foo', '<to be escaped>');
		$this->assertEquals('<tag foo="&lt;to be escaped&gt;" />', $tagBuilder->render());
	}

	/**
	 * @test
	 */
	public function attributeValuesAreNotEscapedIfDisabled() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder('tag');
		$tagBuilder->addAttribute('foo', '<not to be escaped>', FALSE);
		$this->assertEquals('<tag foo="<not to be escaped>" />', $tagBuilder->render());
	}

	/**
	 * @test
	 */
	public function attributesCanBeRemoved() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder('tag');
		$tagBuilder->addAttribute('attribute1', 'attribute1value');
		$tagBuilder->addAttribute('attribute2', 'attribute2value');
		$tagBuilder->addAttribute('attribute3', 'attribute3value');
		$tagBuilder->removeAttribute('attribute2');
		$this->assertEquals('<tag attribute1="attribute1value" attribute3="attribute3value" />', $tagBuilder->render());
	}

	/**
	 * @test
	 */
	public function resetResetsTagBuilder() {
		$tagBuilder = $this->getAccessibleMock('Tx_Fluid_Core_ViewHelper_TagBuilder', array('dummy'));
		$tagBuilder->setTagName('tagName');
		$tagBuilder->setContent('some content');
		$tagBuilder->forceClosingTag(TRUE);
		$tagBuilder->addAttribute('attribute1', 'attribute1value');
		$tagBuilder->addAttribute('attribute2', 'attribute2value');
		$tagBuilder->reset();

		$this->assertEquals('', $tagBuilder->_get('tagName'));
		$this->assertEquals('', $tagBuilder->_get('content'));
		$this->assertEquals(array(), $tagBuilder->_get('attributes'));
		$this->assertFalse($tagBuilder->_get('forceClosingTag'));
	}

	/**
	 * @test
	 */
	public function tagNameCanBeOverridden() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder('foo');
		$tagBuilder->setTagName('bar');
		$this->assertEquals('<bar />', $tagBuilder->render());
	}

	/**
	 * @test
	 */
	public function tagContentCanBeOverridden() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder('foo', 'some content');
		$tagBuilder->setContent('');
		$this->assertEquals('<foo />', $tagBuilder->render());
	}

	/**
	 * @test
	 */
	public function tagIsNotRenderedIfTagNameIsEmpty() {
		$tagBuilder = new Tx_Fluid_Core_ViewHelper_TagBuilder('foo');
		$tagBuilder->setTagName('');
		$this->assertEquals('', $tagBuilder->render());
	}
}

?>
